/*
Author: Radek Henys (admin@mouseviator.com)
Version: 1.0b
Last update: 26.3.2014
License: LGPL v3
*/

#include "common.h"
#include "EnumWindowProcInfo.h"
#include "window.h"

/*
	Helper function to find windows based on process they belong to, window text and if they are toplevel windows or not
*/
BOOL CALLBACK EnumWindowsProc(HWND hWnd, LPARAM lParam) {
	DWORD processId = 0;
	boolean hasMatch;
	EnumWindowProcInfo* info = (EnumWindowProcInfo*)lParam;

	//init hasMatch value
	//at least one search criteria must be set in order to get a match
	//if no one is set, the hasMatch will be false and this cannot be changed by any further code
	hasMatch = (info->m_processID != INVALID_PROCESSID_VALUE) || (info->windowText != NULL);

	//check if process id was passed and get id of process window belongs to
	if (info->m_processID != INVALID_PROCESSID_VALUE) {
		//get id of process the window belongs to
		GetWindowThreadProcessId(hWnd, &processId);
		//check if those match
		hasMatch &= (processId == (DWORD)info->m_processID);
	}

	//check if window text matches
	if (info->windowText != NULL) {
		std::string windowText = getWindowText(hWnd);
		//check for text match
		hasMatch &= (windowText.find(info->windowText) != std::string::npos);
	}

	//check if only top level
	if (info->m_enumType == WE_TOPLEVELONLY) {
		HWND parent = GetParent(hWnd);
		hasMatch &= (parent == NULL);
	}

	if (hasMatch) {
		info->m_windowHandles.push_back(hWnd);
	}
	
	//always return true, we want to enumerate all windows
	return TRUE;
}

/*
	This function returns handles for windows owned by process specified by given process id.

	Parameters:
		processID - id of process we want to find window handles for.
		toplevelonly - Whether the window must be a top level window (not a child window). 
					Supported constants are: WE_ALL  for all windows, WE_TOPLEVELONLY  for top-level windows only. 
					Default is WE_ALL.
	Returns:
		An array of window handles belonging to given process.
	Throws:
		Will throw error if something goes wrong.
*/
int getWindowHandles(lua_State *L) {
	long argument;
	EnumWindowProcInfo info;

	int arg_count = lua_gettop(L);

	//check if given parameter is number
	argument = luaL_checklong(L, 1);
	if (argument < 0) {
		return luaL_error(L, "Invalid argument. A process id given: %d. Process id cannot be negative!", argument);
	}
	else {
		info.m_processID = argument;
	}

	if (arg_count == 2) {
		argument = luaL_checkinteger(L, 2);
		info.m_enumType = (byte)argument;
	}

	BOOL result = EnumWindows(&EnumWindowsProc, (LPARAM)&info);
	if (result == FALSE) {
		return getLastError(L);
	}

	//the info object should contain matching hwnd
	lua_newtable(L);
	//store founded hWnds to lua stack
	for (unsigned int i = 0; i < info.m_windowHandles.size(); i++) {
		lua_pushinteger(L, (long)(info.m_windowHandles[i]));
		lua_rawseti(L, -2, (i + 1));
	}

	return 1;
}


/*
	This function returns text in the title bar for window specified by handle.

	Parameters:
		hWnd - handle of window we want to get size for.
	Returns:
		A text in the title bar of specified window.
	Throws:
		Will throw error if something goes wrong.
*/
int getWindowText(lua_State* L) {
	long hWnd;

	//check if given parameter is number
	hWnd = luaL_checklong(L, 1);

	std::string windowText = getWindowText((HWND)(LONG_PTR)hWnd);
	if (windowText.length() > 0) {
		lua_pushstring(L, windowText.c_str());
	}
	else {
		lua_pushnil(L);
	}

	return 1;
}


/*
	This function tries to find windows specified by text in its title bar. Optionally, the matching 	
	criteria may be tightened by specifying also ID of process the window must belong to and 	
	whether the window can be any window or top level only (window that is not child of any 	
	other window  has no parent).

	Parameters:
		windowText - the text of the window title bar that must be matched (partially)
		toplevelonly - WE_ALL - for all windows, WE_TOPLEVELONLY - toplevel windows only.
	Returns:
		An array of window handles belonging to given process.
	Throws:
		Will throw error if something goes wrong.
*/
int findWindow(lua_State* L) {
	EnumWindowProcInfo info;

	int arg_count = lua_gettop(L);
	//we expect a string
	info.windowText = luaL_checkstring(L, 1);

	//check how much arguments was given
	//2 - window text, bTopLevelOnly
	//3 - window text, processID, bTopLevelOnly
	if (arg_count == 3) {
		long argument = luaL_checklong(L, 2);
		if (argument < 0) {
			return luaL_error(L, "Invalid argument. A process id given: %d. Process id cannot be negative!", argument);
		}
		else {
			info.m_processID = argument;
		}

		//get if only top level windows
		argument = luaL_checkinteger(L, 3);
		info.m_enumType = (byte)argument;
	}
	else if (arg_count == 2) {
		//get if only top level windows
		long argument = luaL_checkinteger(L, 2);
		info.m_enumType = (byte)argument;
	}

	BOOL result = EnumWindows(&EnumWindowsProc, (LPARAM)&info);
	if (result == FALSE) {
		return getLastError(L);
	}

	//the info object should contain matching hwnd
	lua_newtable(L);
	//store founded hWnds to lua stack
	for (unsigned int i = 0; i < info.m_windowHandles.size(); i++) {
		lua_pushinteger(L, (long)(info.m_windowHandles[i]));
		lua_rawseti(L, -2, (i + 1));
	}

	return 1;
}

/*
	This function returns size of window specified by its handle.

	Parameters:
		hWnd - handle of window we want to get size for.
	Returns:
		Width and height of the window.
	Throws:
		Will throw error if something goes wrong.
*/
int getWindowSize(lua_State* L) {
	long hWnd;
	RECT rcWindow;

	//check if given parameter is number
	hWnd = luaL_checklong(L, 1);

	BOOL result = GetWindowRect((HWND)(LONG_PTR)hWnd, &rcWindow);
	if (result == FALSE) {
		return getLastError(L);
	}

	//the calculation works even for negative values:
	//example: rcWindow.right - rcWindow.left
	//				512       -	   -256			=  768
	//				-256      -    -1024		=  768
	lua_pushinteger(L, (rcWindow.right - rcWindow.left));
	lua_pushinteger(L, (rcWindow.bottom - rcWindow.top));

	return 2;
}

/*
	This function returns client size of window specified by its handle.

	Parameters:
		hWnd - handle of window we want to get size for.
	Returns:
		Width and height of the window.
	Throws:
		Will throw error if something goes wrong.
*/
int getClientSize(lua_State *L) {
	long hWnd;
	RECT rcClient;

	//check if given parameter is number
	hWnd = luaL_checklong(L, 1);

	BOOL result = GetClientRect((HWND)(LONG_PTR)hWnd, &rcClient);
	if (result == FALSE) {
		return getLastError(L);
	}

	//the calculation works even for negative values:
	//example: rcWindow.right - rcWindow.left
	//				512       -	   -256			=  768
	//				-256      -    -1024		=  768
	lua_pushinteger(L, (rcClient.right - rcClient.left));
	lua_pushinteger(L, (rcClient.bottom - rcClient.top));

	return 2;
}

/*
	This function moves and optionally resizes window specified by its handle.

	Parameters:
		hWnd - handle of window we want to get size for.
		x - x position of top left corner.
		y - y position of top left corner.
		width - [Optional] width of the window
		height - [Optional] height of the window
	Returns:
		Nothing if no errors, otherwise, exception is thrown.
	Throws:
		Will throw error if something goes wrong.

	TODO: pridat moznost monitoru
*/
int moveWindow(lua_State* L) {
	long hWnd;
	int x, y, width, height;

	int arg_count = lua_gettop(L);
	if (arg_count < 3) {
		return luaL_error(L, "Invalid argument count. 3 to 5 arguments are required!");
	}

	//first argument should be hWnd
	hWnd = luaL_checklong(L, 1);

	//second argument should be x
	x = luaL_checkint(L, 2);

	//thrird argument should be y
	y = luaL_checkint(L, 3);

	if (arg_count > 3) {
		//than we expect width and height
		width = luaL_checkint(L, 4);
		height = luaL_checkint(L, 5);
	}
	else {
		//we need to get current window size
		RECT rcWindow;
		if (GetWindowRect((HWND)(LONG_PTR)hWnd, &rcWindow)) {
			width = (int)(rcWindow.right - rcWindow.left);
			height = (int)(rcWindow.bottom - rcWindow.top);
		}
		else {
			return getLastError(L);
		}
	}

	//move the window
	BOOL result = MoveWindow((HWND)(LONG_PTR)hWnd, x, y, width, height, TRUE);
	if (result == FALSE) {
		return getLastError(L);
	}

	return 0;
}

/*
	This function is simple wrapper for windows function: SetWindowPos

	Parameters:
		hWnd - handle of window we want to get size for.
		hWndInsertAfter - handle of window we want to move this window before in z order.
		x - x position of top left corner.
		y - y position of top left corner.
		width - width of the window
		height - height of the window
		uFlags - flags. See: http://msdn.microsoft.com/en-us/library/windows/desktop/ms633545(v=vs.85).aspx for details.
	Returns:
		Nothing if no errors, otherwise, exception is thrown.
	Throws:
		Will throw error if something goes wrong.

TODO: pridat moznost monitoru
*/
int setWindowPos(lua_State *L) {
	long hWnd;
	long hWndInsertAfter;
	int x, y, width, height;
	int argument;
	UINT uFlags = SWP_NOACTIVATE | SWP_NOZORDER;

	int arg_count = lua_gettop(L);
	if (arg_count < 6) {
		return luaL_error(L, "Invalid argument count. 6 to 7 arguments are required!");
	}

	//first argument should be hWnd
	hWnd = luaL_checklong(L, 1);

	//second parameter is hWndInsertAfter
	hWndInsertAfter = luaL_checklong(L, 2);

	//third argument should be x
	x = luaL_checkint(L, 3);

	//fourth argument should be y
	y = luaL_checkint(L, 4);

	//fifth argument is width
	width = luaL_checkint(L, 5);

	//sixth argument is height
	height = luaL_checkint(L, 6);

	//last argument are flags
	if (arg_count == 7) {
		argument = luaL_checkint(L, 7);
		if (argument > 0) {
			uFlags = (UINT)argument;
		}
	}

	//set window position
	BOOL result = SetWindowPos((HWND)(LONG_PTR)hWnd, (HWND)(LONG_PTR)hWndInsertAfter, x, y, width, height, uFlags);
	if (result == FALSE) {
		return getLastError(L);
	}

	return 0;
}

/*
	This function brings selected window to foreground.

	Parameters:
		hWnd - handle of window we want to bring foreground.
	Returns:
		Nothing if no errors, otherwise, exception is thrown.
	Throws:
		Will throw error if something goes wrong.
*/
int setForeground(lua_State* L) {
	long hWnd;

	//check if given parameter is number
	hWnd = luaL_checklong(L, 1);

	BOOL result = SetForegroundWindow((HWND)(LONG_PTR)hWnd);
	if (result == FALSE) {
		return getLastError(L);
	}

	return 0;
}

/*
	This function tests whether given window is foreground window.

	Parameters:
		hWnd - handle of window we want to check that is foreground.
	Returns:
		True if given window is at foreground.
	Throws:
		Will throw error if something goes wrong.
*/
int isForeground(lua_State* L) {
	long hWnd;

	//check if given parameter is number
	hWnd = luaL_checklong(L, 1);

	HWND hWndForeground = GetForegroundWindow();
	lua_pushboolean(L, (hWnd == (long)hWndForeground));

	return 1;
}

/*
	This function restores the given window.

	Parameters:
		hWnd - handle of window we want to restore.
	Returns:
		Nothing if no errors, otherwise, exception is thrown.
	Throws:
		Will throw error if something goes wrong.
*/
int restoreWindow(lua_State* L) {
	long hWnd;

	//check if given parameter is number
	hWnd = luaL_checklong(L, 1);

	//restore the window
	ShowWindow((HWND)(LONG_PTR)hWnd, SW_RESTORE);

	return 0;
}

/*
	This function minimizes the given window.

	Parameters:
		hWnd - handle of window we want to minimize.
	Returns:
		Nothing if no errors, otherwise, exception is thrown.
	Throws:
		Will throw error if something goes wrong.
*/
int minimizeWindow(lua_State* L) {
	long hWnd;

	//check if given parameter is number
	hWnd = luaL_checklong(L, 1);

	//restore the window
	ShowWindow((HWND)(LONG_PTR)hWnd, SW_MINIMIZE);

	return 0;
}

/*
	This function maximizes the given window.

	Parameters:
		hWnd - handle of window we want to maximize.
	Returns:
		Nothing if no errors, otherwise, exception is thrown.
	Throws:
		Will throw error if something goes wrong.
*/
int maximizeWindow(lua_State* L) {
	long hWnd;

	//check if given parameter is number
	hWnd = luaL_checklong(L, 1);

	//restore the window
	ShowWindow((HWND)(LONG_PTR)hWnd, SW_MAXIMIZE);

	return 0;
}

/*
	This function tests whether the given window is in full screen mode  if it's size is the same or greater than the size of the monitor the window is displayed on.

	Parameters:
		hWnd - handle of window we want to test.
	Returns:
		True if window size is the sam or greater than the size of the monitor it is displayed on, otherwise false.
	Throws:
		Will throw error if something goes wrong.
*/
int isFullScreen(lua_State *L) {
	long hWnd;
	boolean isfullscreen = false;

	//check if given parameter is number
	hWnd = luaL_checklong(L, 1);
	
	//get the id of the monitor the window is on
	HMONITOR hmonitor = MonitorFromWindow((HWND)(LONG_PTR)hWnd, MONITOR_DEFAULTTONULL);
	if (hmonitor) {
		//get monitor info
		MONITORINFO mInfo;
		mInfo.cbSize = sizeof(MONITORINFO);
		RECT rcWindow;
		if (GetMonitorInfo(hmonitor, &mInfo) &&
			GetWindowRect((HWND)(LONG_PTR)hWnd, &rcWindow)) {
			//compare monitor display rect and window rect
			POINT windowSize;
			windowSize.x = rcWindow.right - rcWindow.left;
			windowSize.y = rcWindow.bottom - rcWindow.top;
			POINT monitorSize;
			monitorSize.x = mInfo.rcMonitor.right - mInfo.rcMonitor.left;
			monitorSize.y = mInfo.rcMonitor.bottom - mInfo.rcMonitor.top;
			isfullscreen = (windowSize.x >= monitorSize.x) && (windowSize.y >= monitorSize.y);
		}
	}
	
	lua_pushboolean(L, isfullscreen);
	return 1;
}

/*
	This function switches given window to full-screen mode. However, this does not do any 	
	true Direct-X exclusive full-screen mode. This function will simply maximize the window and
	remove its borders, so it fills the whole screen.

	Parameters:
		hWnd - handle of window we want to get focus to.
	Returns:
		Nothing if no errors, otherwise, exception is thrown.
	Throws:
		Will throw error if something goes wrong.
*/
int setFullScreen(lua_State *L) {
	long hWnd;

	//check if given parameter is number
	hWnd = luaL_checklong(L, 1);
	
	BOOL result = SetWindowLongPtr((HWND)(LONG_PTR)hWnd, GWL_STYLE, WS_POPUP | WS_SYSMENU | WS_VISIBLE);
	if (result == FALSE) {
		return getLastError(L);
	}

	//SetWindowPos needs to be called after SetWindowLongPtr to make sure changes are applied
	result = SetWindowPos((HWND)(LONG_PTR)hWnd, NULL, 0, 0, 0, 0, SWP_FRAMECHANGED | SWP_NOMOVE | SWP_NOSIZE | SWP_NOACTIVATE);
	if (result == FALSE) {
		return getLastError(L);
	}

	//maximize the window
	ShowWindow((HWND)(LONG_PTR)hWnd, SW_MAXIMIZE);
	
	return 0;
}

/*
	This function switches given window to windowed mode. It changes window style so it has 	
	title bar and border. You will notice no change if the window already has a border of course. 	
	This function can be used for example to return normal windowed style to window 	
	maximized with window.setfullscreen.

	Parameters:
		hWnd - handle of window we want to have border.
	Returns:
		Nothing if no errors, otherwise, exception is thrown.
	Throws:
		Will throw error if something goes wrong.
*/
int setWindowed(lua_State *L) {
	long hWnd;

	//check if given parameter is number
	hWnd = luaL_checklong(L, 1);

	//restore window
	ShowWindow((HWND)(LONG_PTR)hWnd, SW_RESTORE);

	//set that the window should have border, minimize, maximize buttons etc - classic window
	BOOL result = SetWindowLongPtr((HWND)(LONG_PTR)hWnd, GWL_STYLE, WS_OVERLAPPEDWINDOW | WS_VISIBLE);
	if (result == FALSE) {
		return getLastError(L);
	}

	//this will apply the changes to window style
	result = SetWindowPos((HWND)(LONG_PTR)hWnd, NULL, 0, 0, 0, 0, SWP_FRAMECHANGED | SWP_NOMOVE | SWP_NOSIZE | SWP_NOACTIVATE);
	if (result == FALSE) {
		return getLastError(L);
	}

	//restore window
	ShowWindow((HWND)(LONG_PTR)hWnd, SW_RESTORE);

	return 0;
}

/*
	This function returns the style of window specified by handle.

	Parameters:
		hWnd - handle of window we want to get style for.
	Returns:
		Window style. See: http://msdn.microsoft.com/en-us/library/windows/desktop/ms632600(v=vs.85).aspx
		for possible values.
	Throws:
		Will throw error if something goes wrong.
*/
int getWindowStyle(lua_State* L) {
	long hWnd;

	//check if given parameter is number
	hWnd = luaL_checklong(L, 1);

	//restore the window
	LONG_PTR result = GetWindowLongPtr((HWND)(LONG_PTR)hWnd, GWL_STYLE);
	if (result == FALSE) {
		return getLastError(L);
	}

	lua_pushinteger(L, result);
	return 1;
}

/*
	This function alters style for window specified by handle.

	Parameters:
		hWnd - handle of window we want to get style for.
		style - the style we want to replace, add or remove by/to/from current style.  See: http://msdn.microsoft.com/en-us/library/windows/desktop/ms632600(v=vs.85).aspx
				for possible values.
		operation - what to do with oldstyle and new style. Values are:
					WSOP_REPLACE - replace old style with new one
					WSOP_ADD - add new style to old style
					WSOP_REMOVE - remove new style from old style.
	Returns:
		Nothing if no errors, otherwise exception is raised.
	Throws:
		Will throw error if something goes wrong.

		TODO: otestovat jednotlive operation
*/
int setWindowStyle(lua_State *L) {
	long hWnd;
	byte operation = WSOP_REPLACE;
	long dwNewStyle, dwOldStyle;
	LONG_PTR result;

	int arg_count = lua_gettop(L);
	//first argument, window handle
	hWnd = luaL_checklong(L, 1);

	//second argument, desired style
	long dwStyle = luaL_checklong(L, 2);

	if (arg_count == 3) {
		//third argument is whether to replace the style or combine with old one
		int argument = luaL_checkint(L, 3);
		if ((argument == WSOP_ADD) || (argument == WSOP_REMOVE)) {
			operation = (byte)argument;
		}
	}

	//get old style
	if (operation != WSOP_REPLACE) {
		result = GetWindowLongPtr((HWND)(LONG_PTR)hWnd, GWL_STYLE);
		if (result == FALSE) {
			return getLastError(L);
		}
		dwOldStyle = (long)result;
	}

	//now what to do?
	switch (operation) {
	case WSOP_ADD:
		dwNewStyle = dwOldStyle | dwStyle;
		break;
	case WSOP_REMOVE:
		dwNewStyle = dwOldStyle ^ dwStyle;
		break;
	default:
		dwNewStyle = dwStyle;
		break;
	}

	result = SetWindowLongPtr((HWND)(LONG_PTR)hWnd, GWL_STYLE, dwNewStyle);
	if (result == FALSE) {
		return getLastError(L);
	}

	//redraw the window
	result = SetWindowPos((HWND)(LONG_PTR)hWnd, NULL, 0, 0, 0, 0, SWP_NOSIZE | SWP_NOMOVE | SWP_NOZORDER | SWP_NOACTIVATE | SWP_FRAMECHANGED);
	if (result == FALSE) {
		return getLastError(L);
	}

	return 0;
}

/*
	This function is direct wrapper for SendMessage.
*/
int sendMessage(lua_State *L) {
	long argument, hWnd;
	UINT iMsg;
	WPARAM wParam;
	LPARAM lParam;

	//first argument is hWnd
	hWnd = luaL_checklong(L, 1);

	//second is message number
	argument = luaL_checklong(L, 2);
	if (argument < 0) {
		return luaL_error(L, "Invalid argument. A message: %d. Message cannot be negative!", argument);
	}
	else {
		iMsg = (UINT)argument;
	}

	//third is wParam
	wParam = luaL_checklong(L, 3);

	//fourth is lParam
	lParam = luaL_checklong(L, 4);

	LRESULT result = SendMessage((HWND)(LONG_PTR)hWnd, iMsg, wParam, lParam);

	lua_pushinteger(L, result);

	return 1;
}